/**
 *       Name: SaveDialog.java
 *    Purpose: Visual display for Export Choices
 *  Copyright: 2000 Regents of the University of California and the
 *             National Center for Ecological Analysis and Synthesis
 *    Authors: @higgins@
 *    Release: @release@
 *
 *   '$Author: leinfelder $'
 *     '$Date: 2008/06/26 21:05:51 $'
 * '$Revision: 1.6 $'
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */
package edu.ucsb.nceas.morpho.first.edml.commands;

import java.awt.BorderLayout;
import java.awt.Component;
import java.awt.Cursor;
import java.awt.Window;
import java.io.File;
import java.io.FileInputStream;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import javax.swing.Box;
import javax.swing.JButton;
import javax.swing.JCheckBox;
import javax.swing.JDialog;
import javax.swing.JLabel;
import javax.swing.JPanel;

import edu.msu.first.dataimport.DataFileColumns;
import edu.msu.first.dataimport.Utilities;
import edu.ucsb.nceas.morpho.Morpho;
import edu.ucsb.nceas.morpho.datapackage.AbstractDataPackage;
import edu.ucsb.nceas.morpho.datapackage.AccessionNumber;
import edu.ucsb.nceas.morpho.datastore.FileSystemDataStore;
import edu.ucsb.nceas.morpho.datastore.MetacatDataStore;
import edu.ucsb.nceas.morpho.datastore.MetacatUploadException;
import edu.ucsb.nceas.morpho.first.edml.AssessmentItemUtil;
import edu.ucsb.nceas.morpho.first.edml.EdMLDataViewContainerPanel;
import edu.ucsb.nceas.morpho.first.plugins.wizard.EdMLWizardSettings;
import edu.ucsb.nceas.morpho.framework.MorphoFrame;
import edu.ucsb.nceas.morpho.framework.UIController;
import edu.ucsb.nceas.morpho.plugins.datapackagewizard.WidgetFactory;
import edu.ucsb.nceas.morpho.util.Log;
import edu.ucsb.nceas.utilities.OrderedMap;

/**
 * A dialog box for user choice of export options
 */
public class SaveDialog extends JDialog {

	/** Control button */
	private JButton executeButton = null;
	private JButton cancelButton = null;

	private boolean showPackageFlag = true;

	/** Radio button */
	private JCheckBox localLoc = new JCheckBox("Save Locally");
	private JCheckBox networkLoc = new JCheckBox("Save to Network.");

	private static final int PADDINGWIDTH = 8;
	private static String WARNING = "Please choose where you would like to save the data package.";

	/** A reference to morpho frame */
	MorphoFrame morphoFrame = null;

	/** A string indicating the morpho frame's type */
	String morphoFrameType = null;

	/** selected docid */
	String selectDocId = null;

	/** flag to indicate selected data package has local copy */
	private boolean inLocal = false;

	/** flag to indicate selected data package has local copy */
	private boolean inNetwork = false;

	/** the AbstractDataPackage object to be saved */
	AbstractDataPackage adp = null;

	/**
	 * Construct a new instance of the dialog where parent is morphoframe
	 * 
	 */
	public SaveDialog(AbstractDataPackage adp) {
		setModal(true);
		this.adp = adp;
		morphoFrame = UIController.getInstance().getCurrentActiveWindow();
		initialize(morphoFrame);

	}

	public SaveDialog(AbstractDataPackage adp, boolean showPackageFlag) {
		this(adp);
		this.showPackageFlag = showPackageFlag;
	}

	/** Method to initialize save dialog */
	private void initialize(Window parent) {
		// Set OpenDialog size depent on parent size
		int parentWidth = parent.getWidth();
		int parentHeight = parent.getHeight();
		int dialogWidth = 400;
		int dialogHeight = 270;
		setSize(dialogWidth, dialogHeight);
		setResizable(false);

		// Set location of dialog, it shared same center of parent
		double parentX = parent.getLocation().getX();
		double parentY = parent.getLocation().getY();
		double centerX = parentX + 0.5 * parentWidth;
		double centerY = parentY + 0.5 * parentHeight;
		int dialogX = (new Double(centerX - 0.5 * dialogWidth)).intValue();
		int dialogY = (new Double(centerY - 0.5 * dialogHeight)).intValue();
		setLocation(dialogX, dialogY);

		setTitle("Save Current DataPackage");
		// Set the default close operation is dispose
		setDefaultCloseOperation(DISPOSE_ON_CLOSE);

		// Set the border layout as layout
		getContentPane().setLayout(new BorderLayout(0, 0));
		// Add padding for left and right
		getContentPane().add(BorderLayout.EAST,
				Box.createHorizontalStrut(PADDINGWIDTH));
		getContentPane().add(BorderLayout.WEST,
				Box.createHorizontalStrut(PADDINGWIDTH));

		// Create JPanel and set it border layout
		JPanel mainPanel = new JPanel();
		mainPanel.setLayout(new BorderLayout(0, 0));

		// Create note box and add it to the north of mainPanel
		Box noteBox = Box.createVerticalBox();
		noteBox.add(Box.createVerticalStrut(PADDINGWIDTH));
		JLabel note = WidgetFactory.makeHTMLLabel(WARNING, 2);
		/*
		 * JTextArea note = new JTextArea(WARNING); note.setEditable(false);
		 * note.setLineWrap(true); note.setWrapStyleWord(true);
		 * note.setOpaque(false);
		 */
		noteBox.add(note);
		noteBox.add(Box.createVerticalStrut(PADDINGWIDTH));
		mainPanel.add(BorderLayout.NORTH, noteBox);

		// Create a radio box
		Box radioBox = Box.createVerticalBox();
		radioBox.add(localLoc);
		radioBox.add(networkLoc);

		// create another center box which will put radion box in the center
		// and it will be add into center of mainPanel
		Box centerBox = Box.createHorizontalBox();
		centerBox.add(Box.createHorizontalGlue());
		centerBox.add(radioBox);
		centerBox.add(Box.createHorizontalGlue());
		mainPanel.add(BorderLayout.CENTER, centerBox);

		// Finish mainPanel and add it the certer of contentpanel
		getContentPane().add(BorderLayout.CENTER, mainPanel);

		// Create bottom box
		Box bottomBox = Box.createVerticalBox();
		getContentPane().add(BorderLayout.SOUTH, bottomBox);
		// Create padding between result panel and Contorl button box
		bottomBox.add(Box.createVerticalStrut(PADDINGWIDTH));
		// Create a controlbuttionBox
		Box controlButtonsBox = Box.createHorizontalBox();
		controlButtonsBox.add(Box.createHorizontalGlue());

		// Save button
		executeButton = new JButton("Save");
		controlButtonsBox.add(executeButton);
		controlButtonsBox.add(Box.createHorizontalStrut(PADDINGWIDTH));

		// Cancel button
		cancelButton = new JButton("Cancel");
		controlButtonsBox.add(cancelButton);
		controlButtonsBox.add(Box.createHorizontalStrut(PADDINGWIDTH));

		// Add controlButtonsBox to bottomBox
		bottomBox.add(controlButtonsBox);
		// Add the margin between controlButtonPanel to the bottom line
		bottomBox.add(Box.createVerticalStrut(10));

		SymAction lSymAction = new SymAction();
		executeButton.addActionListener(lSymAction);
		cancelButton.addActionListener(lSymAction);

		String location = adp.getLocation();
		if (location.equals("")) { // never been saved
			localLoc.setEnabled(true);
			networkLoc.setEnabled(true);
			localLoc.setSelected(true);
			networkLoc.setSelected(false);
		} else if (location.equals(AbstractDataPackage.LOCAL)) {
			localLoc.setEnabled(false);
			networkLoc.setEnabled(true);
			localLoc.setSelected(true);
			networkLoc.setSelected(true);
		} else if (location.equals(AbstractDataPackage.METACAT)) {
			localLoc.setEnabled(true);
			networkLoc.setEnabled(false);
			localLoc.setSelected(true);
			networkLoc.setSelected(true);
		} else if (location.equals(AbstractDataPackage.BOTH)) {
			localLoc.setEnabled(false);
			networkLoc.setEnabled(false);
			localLoc.setSelected(false);
			networkLoc.setSelected(false);
		}

		setVisible(true);

	}

	/** Method to enable executeButton and assign command */
	private void enableExecuteButton(Object object, JDialog dialog) {
	}// enableExecuteButton

	class SymAction implements java.awt.event.ActionListener {
		public void actionPerformed(java.awt.event.ActionEvent event) {
			Object object = event.getSource();
			if (object == executeButton) {
				executeButton_actionPerformed(event);
			} else if (object == cancelButton) {
				cancelButton_actionPerformed(event);
			}
		}
	}

	void cancelButton_actionPerformed(java.awt.event.ActionEvent event) {
		this.setVisible(false);
		this.dispose();
	}

	void executeButton_actionPerformed(java.awt.event.ActionEvent event) {
		
		//set the hourglass
		setCursor(new Cursor(Cursor.WAIT_CURSOR));

		//save the manual data changes from the table (if any)
		Component comp = morphoFrame.getContentComponent();
		if (comp instanceof EdMLDataViewContainerPanel) {
			EdMLDataViewContainerPanel edvcp = (EdMLDataViewContainerPanel) comp;
			edvcp.saveDataChanges();
		}
		
		//set the new accession number for the DP
		boolean problem = false;
		Morpho morpho = Morpho.thisStaticInstance;
		AccessionNumber an = new AccessionNumber(morpho);
		String location = adp.getLocation();
		String id = adp.getAccessionNumber();
		if (location.equals("")) {
			try {				
				if (id.indexOf("temporary") > -1) {
					String nextid = an.getNextId();
					adp.setAccessionNumber(nextid);
				} else {
					String newid = an.incRev(id);
					adp.setAccessionNumber(newid);
				}
			} catch (Exception www) {
				// no valid accession number; thus create one
				String nextid = an.getNextId();
				adp.setAccessionNumber(nextid);
			}
		}

		//serialize it all now
		try {
			if ((localLoc.isSelected()) && (localLoc.isEnabled())
					&& (networkLoc.isSelected()) && (networkLoc.isEnabled())) {
				this.saveAll(AbstractDataPackage.BOTH);
				this.updateDataFile(AbstractDataPackage.BOTH);
				adp.serialize(AbstractDataPackage.BOTH);
				adp.setLocation(AbstractDataPackage.BOTH);
				adp.serializeData();
			} else if ((localLoc.isSelected()) && (localLoc.isEnabled())) {
				this.saveAll(AbstractDataPackage.LOCAL);
				this.updateDataFile(AbstractDataPackage.LOCAL);
				adp.serialize(AbstractDataPackage.LOCAL);
				adp.setLocation(AbstractDataPackage.LOCAL);
				adp.serializeData();
			} else if ((networkLoc.isSelected()) && (networkLoc.isEnabled())) {
				this.saveAll(AbstractDataPackage.METACAT);
				this.updateDataFile(AbstractDataPackage.METACAT);
				adp.serialize(AbstractDataPackage.METACAT);
				adp.setLocation(AbstractDataPackage.METACAT);
				adp.serializeData();
			} else {
				Log.debug(1, "No location for saving is selected!");
			}
		} catch (MetacatUploadException mue) {
			
			//set back to normal since these might display messages
			setCursor(new Cursor(Cursor.DEFAULT_CURSOR));
			
			String errormsg = mue.getMessage();
			if (errormsg.indexOf("ERROR SAVING DATA TO METACAT") > -1) {
				// error in saving data file
				Log.debug(5, "Problem Saving Data to Metacat");
			} else if (errormsg.indexOf("is already in use") > -1) {
				// metadata insert error
				Log.debug(5, "Problem Saving Data: Id already in use");
			} else if (errormsg.indexOf("Document not found for Accession number") > -1) {
				// error in updating data file
				Log.debug(5,"Problem Saving Data: Document not found for Accession number");
			} else if (errormsg.indexOf("Invalid content") > -1) {
				// validation error
				Log.debug(5, "Problem Saving Data due to invalid content");
			}

			Log.debug(20, "Problem Saving\n" + mue.getMessage());
			problem = true;
		}

		this.setVisible(false);
		this.dispose();

		if (!problem) {
			if (showPackageFlag) {
				UIController.showNewPackageNoLocChange(adp);
			} else {
				MorphoFrame morphoFrame = 
					UIController.getInstance().getCurrentActiveWindow();
				morphoFrame.setVisible(false);
				UIController controller = UIController.getInstance();
				controller.removeWindow(morphoFrame);
				morphoFrame.dispose();
			}
		}
		setCursor(new Cursor(Cursor.DEFAULT_CURSOR));

	}
	
	private void updateDataFile(String location) {

		//set the new accession number in the data file if it exists
		if (adp.getEntityCount() > 0) {
			AccessionNumber an = new AccessionNumber(Morpho.thisStaticInstance);
			String distributionUrl = adp.getDistributionUrl(0, 0, 0);
			String originalDataFileId = distributionUrl.substring(distributionUrl.lastIndexOf("/")+1);
			String updatedDataFileId = an.incRev(originalDataFileId);
			File originalDataFile = null;
			File updatedDataFile = null;
			FileSystemDataStore fds = new FileSystemDataStore(Morpho.thisStaticInstance);
			MetacatDataStore mds = new MetacatDataStore(Morpho.thisStaticInstance);
	
			try {
				//get the original file
				if (location.equals(AbstractDataPackage.LOCAL) || location.equals(AbstractDataPackage.BOTH)) {
					originalDataFile = fds.openFile(originalDataFileId);
				}
				else {
					originalDataFile = mds.openDataFile(originalDataFileId);
				}
				
				File assessmentTempFile = File.createTempFile("assessmentIdData", ".tmp");
				File tempFile = File.createTempFile("itemIdData", ".tmp");
				
				//replace the old id with new id
				Utilities.replaceColumnValues(
						originalDataFile, 
						assessmentTempFile, 
						DataFileColumns.ASSESSMENT_IDENTIFIER, 
						adp.getAccessionNumber());
				
				//replace any questionIds that have changed
				Map replacementIdMap = new HashMap();
				List assessmentItemIds = AssessmentItemUtil.getAssessmentItemIds(adp);
				Iterator idIter = assessmentItemIds.iterator();
				while (idIter.hasNext()) {
					String id = (String) idIter.next();
					String idWithoutRev = an.getIdNoRev(id);
					replacementIdMap.put(idWithoutRev, id);
				}
				
				//replace the old ids with new ids
				Utilities.replaceIdentifiers(
						assessmentTempFile, 
						tempFile, 
						DataFileColumns.QUESTION_IDENTIFIER, 
						replacementIdMap,
						true);
	
				//make the new data file in the local directory
				if (location.equals(AbstractDataPackage.LOCAL) || location.equals(AbstractDataPackage.BOTH)) {
					updatedDataFile = fds.newDataFile(updatedDataFileId, new FileInputStream(tempFile));
				}
				if (location.equals(AbstractDataPackage.METACAT) || location.equals(AbstractDataPackage.BOTH)) {
					mds.newDataFile(updatedDataFileId, tempFile);
					updatedDataFile = tempFile;
				}
								
				//set the new data file information
				adp.setDistributionUrl(0, 0, 0, EdMLWizardSettings.URN_ROOT + updatedDataFileId);
				adp.setPhysicalSize(0, 0, String.valueOf(updatedDataFile.length()));
				
				tempFile.delete();
				
			} catch (Exception e) {
				Log.debug(5, "Error encountered while synchronizing data with metadata: \n" + e.getMessage());
				e.printStackTrace();
			}
		}
	}

	private boolean saveAll(String location) {
		//get the list of assessment items from current adp
		List assessmentItems = AssessmentItemUtil.getAssessmentItems(adp);
		
		//save them, but do not increment the ids right now
		// the parser has already saved the item locally, so this is mostly for getting them to metacat
		//and we are not editing items yet, so the rev will always be ".1"
		//TODO: handle fringe cases when edits have been performed on the item and the rev is different on server/local
		assessmentItems = AssessmentItemUtil.saveDataPackages(assessmentItems, location, false);
		
		//get the refreshed map of ids (revision numbers likely changed)
		OrderedMap map = AssessmentItemUtil.dataPackages2Map(assessmentItems);
		
		//re-insert those into the assesssment datapackage
		return AssessmentItemUtil.insertAssessmentItems(map, adp);
		
	}
}// SaveDialog
