package edu.msu.first.parser.gui.undo;

import javax.swing.undo.*;

import edu.msu.first.parser.gui.QuestionComponent;

/**
 * UndoableEdit for when the user starts or ends "edit mode" on a particular question.
 * This UndoableEdit is not significant, meaning it does not show up in the undo menu.
 * Rather, it is automatically undone if the user undoes his/her actions to a point in
 * time before this happened.
 * @author Garrison Benson
 *
 */
public class EditModeUndoableEdit extends AbstractUndoableEdit {
	
	private boolean oldMode;
	private boolean newMode;
	private QuestionComponent questionComponent;
	
	/**
	 * Constructor for EditModeUndoableEdit
	 * @param question The <code>Question</code> for which the user has clicked either "Edit" or "Done".
	 * @param oldMode <code>true</code> if the user started in Edit mode, <code>false</code> otherwise
	 * @param newMode <code>true</code> if the user ended in Edit mode, <code>false</code> otherwise
	 */
	public EditModeUndoableEdit(QuestionComponent questionComponent, boolean oldMode, boolean newMode) {
		this.questionComponent = questionComponent;
		this.oldMode = oldMode;
		this.newMode = newMode;
	}
	
	@Override
	public String getPresentationName() {
		return "edit mode";
	}
	
	@Override
	public boolean isSignificant() {
		return false;
	}
	
	@Override
	public void undo() throws CannotUndoException {
		questionComponent.setEditMode(oldMode);
		super.undo();
	}
	
	@Override
	public void redo() throws CannotRedoException {
		questionComponent.setEditMode(newMode);
		super.redo();
	}
	
	public QuestionComponent getQuestionComponent() {
		return questionComponent;
	}
	
	@Override
	public boolean addEdit(UndoableEdit anEdit) {
		
		// TODO: Find some way to throw the edit away if the newMode and oldMode are the same.
		
		boolean retVal = false;
		// If the other edit is the same type of edit and it affects the same QuestionComponent...
		if ((anEdit.getClass() == getClass()) &&
				((EditModeUndoableEdit)anEdit).getQuestionComponent() == getQuestionComponent()) {
			
			// If the new edit's oldMode is the same as our newMode (in other words,
			// if the edits are in chronological order)...
			if (((EditModeUndoableEdit)anEdit).oldMode == newMode) {
				newMode = ((EditModeUndoableEdit)anEdit).newMode;
				retVal = true;
			}
		}
		
		
		
		return retVal;
	}
	
}
