package edu.msu.first.parser.gui.wizard;

import java.awt.Component;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.event.MouseAdapter;
import java.awt.event.MouseEvent;
import java.awt.event.MouseListener;
import java.awt.image.BufferedImage;
import java.io.File;
import java.util.ArrayList;
import java.util.LinkedList;
import java.util.List;
import java.util.Set;

import javax.swing.JComponent;
import javax.swing.JLabel;
import javax.swing.JList;
import javax.swing.JMenuItem;
import javax.swing.JOptionPane;
import javax.swing.JPopupMenu;
import javax.swing.JScrollPane;
import javax.swing.event.ListSelectionEvent;
import javax.swing.event.ListSelectionListener;

import net.miginfocom.swing.MigLayout;
import edu.msu.first.dataimport.gui.BasePanel;
import edu.msu.first.parser.assessment.ImageOnPage;
import edu.msu.first.parser.extract.PDFExtract;
import edu.msu.first.parser.gui.ImageList;
import edu.msu.first.parser.gui.ImagePane;
import edu.msu.first.parser.gui.ScreenshotListener;
import edu.msu.first.parser.gui.SelectionEvent;
import edu.msu.first.parser.gui.Selector;
import edu.msu.first.wizard.WizardData;

public class VectorImageRasterization extends PanelWithPDFPane {

	public enum ComponentNames {
		LIST_PAGES_WITH_VECTORS, SCROLLPANE_PAGES_WITH_VECTORS,
		LABEL_PAGES_WITH_VECTORS, LABEL_PDF_PANE, LABEL_RASTER_IMAGE_PANE,
		LABEL_VECTOR_IMAGE_PAGE,
		MENU_DELETE_IMAGE
	}
	
	/**
	 * This class extends the functionality of the ScreenshotListener class so that 
	 * once the selection event has fired, the image object is retrieved from the
	 * ScreenshotListener, saved to a file, and then maintained in the rasterized 
	 * images list
	 * @author Ryan McFall
	 *
	 */
	public class SaveScreenShotImageListener extends ScreenshotListener {
		int currentImageNumber;
		
		public SaveScreenShotImageListener (JComponent target) {
			super (target);
			currentImageNumber = 0;
		}

		@Override
		public void selectionPerformed(SelectionEvent e) {			
			super.selectionPerformed(e);			
			BufferedImage image = getImage();
			String filename = BasePanel.getMorphoDir() + File.separator + "vectorimage_" + currentImageNumber + ".jpg";
			File imageFile = new File (filename);
			assert (!imageFile.exists());
			
			try {
				writeFile(imageFile);				
				Component newComponent = vectorImagePane.addImage(filename);
				newComponent.addMouseListener(popupListener);
								
				ImageOnPage newImage = new ImageOnPage (filename, pdfPane.getCurrentPageNumber(), e.getRect());
				rasterizedImages.add(newImage);
				
				currentImageNumber++;
				if (logger.isDebugEnabled()) {
					logger.debug ("Wrote rasterized version of vector image to file " + filename);
				}
				selector.clearSelection();
				pdfPane.repaint();
			}
			catch (Exception ex) {
				logger.error("Exception occurred trying to save rasterized version of vector image to " + filename);
				JOptionPane.showMessageDialog(VectorImageRasterization.this, "An exception occurred trying to rasterize the vector image", "Cannot save image", JOptionPane.ERROR_MESSAGE);
			}			
		}				
	}

	protected static final String SELECTED_LABEL = "SELECTED_LABEL";
	
	protected JLabel lblPagesWithVectors;
	protected JLabel lblPDFPane;
	protected JLabel lblRasterImagePane;
	protected JLabel lblVectorImagePane;
	
	protected JList listPagesWithVectors;
	protected JScrollPane scpPagesWithVectors;
	protected ImagePane rasterImagePane;
	protected ImagePane vectorImagePane;
	
	private Selector selector;
	
	private List<ImageOnPage> rasterizedImages;
	private PDFExtract extracter;
	
	private JPopupMenu popupMenu;
	private JMenuItem menuItemDeleteImage;
	protected MouseListener popupListener;
	
	public VectorImageRasterization (WizardData data) {
		super (data, true);								
	}
	
	@Override
	protected void createPanelGUI() {
		super.createPanelGUI();
		
		if (wizardData.isPropertySet(WizardPropertyNames.RASTERIZED_IMAGES)) {
			rasterizedImages = (List<ImageOnPage>) wizardData.getProperty(WizardPropertyNames.RASTERIZED_IMAGES);
		}
		else {
			rasterizedImages = new ArrayList<ImageOnPage> ();
		}	
		
		File pdfFile = (File) wizardData.getProperty(WizardPropertyNames.PDF_FILE);
		
		extracter = (PDFExtract) wizardData.getProperty(WizardPropertyNames.PDF_EXTRACT);
		
		LinkedList<String> rasterImages = extracter.getImages();
				
		if (logger.isDebugEnabled()) {
			logger.debug ("Found " + rasterImages.size() + " images in the document " + pdfFile.getAbsolutePath());
		}
		ImageList imageList = new ImageList (rasterImages);
		rasterImagePane = new ImagePane ();
		rasterImagePane.setImageList(imageList);
			
		vectorImagePane = new ImagePane ();
		
		lblPagesWithVectors = factory.createJLabel (ComponentNames.LABEL_PAGES_WITH_VECTORS, "Pages containing vector images");
		lblPDFPane = factory.createJLabel (ComponentNames.LABEL_PDF_PANE, "Assessment Preview");
		lblRasterImagePane = factory.createJLabel (ComponentNames.LABEL_RASTER_IMAGE_PANE, "Non-vector images on this page");
		lblVectorImagePane = factory.createJLabel (ComponentNames.LABEL_VECTOR_IMAGE_PAGE, "Vector images on this page");
		listPagesWithVectors = factory.createJList(ComponentNames.LIST_PAGES_WITH_VECTORS, getPagesContainingVectorImages().toArray());
		scpPagesWithVectors = factory.createJScrollPane(ComponentNames.SCROLLPANE_PAGES_WITH_VECTORS, listPagesWithVectors);
		
		MigLayout layout = (MigLayout) bodyPanel.getLayout();
		layout.setLayoutConstraints("flowy");		
		bodyPanel.add(lblPagesWithVectors);
		bodyPanel.add(scpPagesWithVectors, "aligny top, spany, growx, growy, wrap");
		
		bodyPanel.add(lblPDFPane);
		bodyPanel.add(pdfPane, "aligny top, spany, wrap");
		
		bodyPanel.add(lblRasterImagePane, "spany, split 4");
		bodyPanel.add(rasterImagePane, "aligny top, growx, growy, sg 1");
		bodyPanel.add(lblVectorImagePane);
		bodyPanel.add(vectorImagePane, "aligny top, growx, growy, wrap, sg 1");
								
		JComponent viewingComponent = (JComponent) pdfPane.getViewingComponent();		
		selector = new Selector (viewingComponent);
		selector.setActive(true);				
		// Set up screen capture
		selector.addSelectionListener(new SaveScreenShotImageListener(viewingComponent));
		
		
		popupMenu = new JPopupMenu ("Image");
		menuItemDeleteImage = factory.createJMenuItem(ComponentNames.MENU_DELETE_IMAGE, "Delete Image");
		popupMenu.add(menuItemDeleteImage);
		setupListeners ();
		
		listPagesWithVectors.setSelectedIndex(0);
	}

	protected void setupListeners () {
		listPagesWithVectors.addListSelectionListener(new ListSelectionListener () {
			public void valueChanged(ListSelectionEvent e) {
				int selectedIndex = (Integer) listPagesWithVectors.getSelectedValue();
				loadPage (selectedIndex);				
			}			
		});
		
		popupListener = new MouseAdapter () {						
			@Override
			public void mouseReleased(MouseEvent event) {
				if (!event.isPopupTrigger()) return;
				popupMenu.show((Component) event.getSource(), event.getX(), event.getY());
				JLabel label = (JLabel) event.getSource();
				//  Store the selected label so that the menuItemDeleteImage method can retrieve it later
				menuItemDeleteImage.putClientProperty(SELECTED_LABEL, label);				
			}					
		};
		
		menuItemDeleteImage.addActionListener(new ActionListener () {
			public void actionPerformed(ActionEvent arg0) {
				JLabel selectedLabel = (JLabel) menuItemDeleteImage.getClientProperty(SELECTED_LABEL);								
				String filename = vectorImagePane.removeImage(selectedLabel);				
				for (int i = 0; i < rasterizedImages.size(); i++) {
					ImageOnPage image = rasterizedImages.get(i);
					if (image.getFilename().equals (filename)) {
						rasterizedImages.remove(i);
						
						//  Delete the file associated with this image
						File toDelete = new File (filename);
						if (!toDelete.delete()) {
							JOptionPane.showMessageDialog(
									VectorImageRasterization.this, 
									"Unable to delete the file associated with the image\n.  If you want to remove it manually, it can be found at " + filename, 
									"Unable to delete file", 
									JOptionPane.INFORMATION_MESSAGE
							);
						}
					}
				}
			}			
		});
	}
	
	protected void loadPage (int pageNumber) {
		pdfPane.jumpToPage(pageNumber);
		
		//  Fill up the raster and vector image panes with images from the new page
		// rasterImagePane.clear();
		vectorImagePane.clear();
		
		for (ImageOnPage image : rasterizedImages) {
			if (image.getPageNum() == pageNumber) {
				Component newComponent = vectorImagePane.addImage(image.getFilename());
				newComponent.addMouseListener(popupListener);
			}
		}		
	}
	
	@Override
	protected String getPanelInstructions() {
		StringBuffer buffer = new StringBuffer (1024);
		buffer.append ("Many PDF files contain vector images.  For example, one very common case is a table.  ");
		buffer.append ("Since it is difficult to identify these vector images automatically, your help is needed ");
		buffer.append (" to identify them.");
		
		buffer.append ("<p>");
		buffer.append ("The list at left contains a list of the pages from your PDF file on which we have found vector images.");
		buffer.append ("For each page, draw a box around the tables, figures and diagrams that don't show up ");
		buffer.append ("in the list of images shown on the right");
		buffer.append ("</p>");
		
		buffer.append ("<p>");
		buffer.append ("When you have identified the vector images on all of the pages, click Next");
		buffer.append ("</p>");
		return buffer.toString();
	}

	public String getPageTitle() {
		return "Identify Vector Images";
	}

	public void saveValues(WizardData data) {
		super.saveValues(data);
		data.setProperty(WizardPropertyNames.RASTERIZED_IMAGES, rasterizedImages);
	}

	/**
	 * This method determines the set of pages in the PDF file that contain
	 * vector images
	 * @return a java.util.List of integers corresponding to the page numbers
	 * in the document containing vector images
	 */
	protected List<Integer> getPagesContainingVectorImages () {		
	    ArrayList<ImageOnPage> vectors=extracter.getVectorLocations();
		List<Integer> pages = new ArrayList<Integer> ();		
		for (ImageOnPage image : vectors) {		
			if (!pages.contains(image.getPageNum())) {
				pages.add (image.getPageNum());
			}
		}
		return pages;
	}

	@Override
	public boolean isFinishEnabledInitially() {
		return true;
	}

	@Override
	public boolean isNextEnabledInitially() {
		return true;
	}

	@Override
	public boolean requiresWaitCursorWhileLoading() {
		return true;
	}	
	
	
}
